//
//  TodoItemEditor.swift
//  Do It
//
//  Created by Jim Dovey on 5/21/20.
//  Copyright © 2020 Pragmatic Programmers. All rights reserved.
//

import SwiftUI

// START:DefiniteOptionals
fileprivate extension TodoItem {
    var hasDueDate: Bool {
        get { date != nil }
        set {
            if newValue && date == nil { //<label id="code.ch4.hasdate"/>
                date = Date()
            }
            else if !newValue {
                date = nil
            }
        }
    }
    
    var definiteDate: Date {
        get { date ?? Date() } //<label id="code.ch4.defaultDate"/>
        set { date = newValue }
    }
    
    var definiteNotes: String {
        get { notes ?? "" }
        set {
            if newValue.isEmpty {
                notes = nil //<label id="code.ch4.empty.notes"/>
            }
            else {
                notes = newValue
            }
        }
    }
}
// END:DefiniteOptionals

// START:EditorView
struct TodoItemEditor: View {
    @Binding var item: TodoItem //<label id="code.ch4.editor.binding"/>
    @EnvironmentObject private var data: DataCenter
    // START:ShowTimeComponent
    @State private var showTime: Bool
    
    init(item: Binding<TodoItem>) {
        self._item = item
        self._showTime = State(wrappedValue: false)
        
        if let date = item.date.wrappedValue {
            let components = Calendar.current.dateComponents(
                [.hour, .minute], from: date)
            self.showTime = components.hour! != 0 || components.minute != 0
        }
    }
    // END:ShowTimeComponent
    
    // START:NotesEditor
    var notesEditor: some View {
        TextView(text: self.$item.definiteNotes)
            .padding(.horizontal)
            .navigationBarTitle("Notes: \(item.title)")
    }
    // END:NotesEditor
    
    var body: some View {
        Form {
            // <literal:elide> form contents will go here </literal:elide>
            // END:EditorView
            // START:TitleField
            TextField("Title", text: $item.title)
            // END:TitleField
            // START:ListPicker
            Picker("List", selection: $item.listID) {
                ForEach(data.todoLists) { list in
                    Text(list.name).tag(list.id)
                }
            }
            // END:ListPicker
            // START:PriorityPicker
            Picker("Priority", selection: $item.priority) {
                ForEach(TodoItem.Priority.allCases, id: \.self) {
                    Text(LocalizedStringKey($0.rawValue.capitalized))
                }
            }
            // END:PriorityPicker
            // START:DueDateCheck
            HStack {
                Text("Due Date")
                Spacer()
                Toggle("Has Due Date", isOn: $item.hasDueDate.animation())
                    .labelsHidden()
            }
            // END:DueDateCheck
            // START:DatePicker
            if self.item.hasDueDate {
                Toggle("Include Time", isOn: $showTime)
                HStack {
                    DatePicker("Due Date", selection: $item.definiteDate,
                               displayedComponents: showTime
                                    ? [.date, .hourAndMinute]
                                    : .date)
                        .datePickerStyle(WheelDatePickerStyle())
                        .labelsHidden()
                        .frame(maxWidth: .infinity, alignment: .center)
                }
            }
            // END:DatePicker
            // START:NotesLink
            NavigationLink("Notes", destination: notesEditor)
            // END:NotesLink
            // START:EditorView
        }
        .navigationBarTitle(Text("Editing: \(item.title)"),
                            displayMode: .inline)
    }
}

struct TodoItemEditor_Previews: PreviewProvider {
    static var previews: some View {
        NavigationView {
            StatefulPreviewWrapper(defaultTodoItems[0]) { //<label id="code.ch4.preview.state.wrapper"/>
                TodoItemEditor(item: $0)
            }
        }
        .environmentObject(DataCenter())
    }
}
// END:EditorView
